<?php
// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}
header("Content-Type: application/json");

session_start();
require_once '../../config/config.php';
include "../functions.php";

// Log function
function logTransaction($message, $type = "info")
{
    $logFile = __DIR__ . "/logs/transaction_log_" . date("Y-m-d") . ".log";
    $logMessage = "[" . date("Y-m-d H:i:s") . "] [$type] $message" . PHP_EOL;
    file_put_contents($logFile, $logMessage, FILE_APPEND);
}

// Check DB connection
if (!isset($conn)) {
    $errorMessage = "Database connection not established";
    logTransaction($errorMessage, "error");
    die(json_encode(["status" => "error", "message" => $errorMessage]));
}

// Allow only POST requests
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    http_response_code(405);
    logTransaction("Method not allowed", "error");
    echo json_encode(["status" => "error", "message" => "Method not allowed"]);
    exit();
}

// Fetch Token
$tokenData = getEversendToken();
if (!$tokenData) {
    logTransaction("Failed to retrieve Eversend token", "error");
    echo json_encode(["status" => "error", "message" => "Authentication failed"]);
    exit();
}

try {
    // Get and decode JSON input
    $json = file_get_contents("php://input");
    $data = json_decode($json, true);

    if (!$data || !is_array($data)) {
        throw new Exception("Invalid JSON data received");
    }

    // Required fields
    $required_fields = ["phone", "amount", "country", "currency", "otp", "customer", "uuid", "service_fee", "charges"];
    foreach ($required_fields as $field) {
        if (!isset($data[$field]) || $data[$field] === "") {
            throw new Exception("$field is required");
        }
    }

    // Sanitize and validate inputs
    $phone = filter_var(trim($data["phone"]), FILTER_SANITIZE_STRING);
    $amount = floatval($data["amount"]);
    $country = filter_var(trim($data["country"]), FILTER_SANITIZE_STRING);
    $currency = filter_var(trim($data["currency"]), FILTER_SANITIZE_STRING);
    $transactionRef = uniqid("txn_", true);
    $otp = $data["otp"];
    $customer = $data["customer"];
    $uuid = filter_var(trim($data["uuid"]), FILTER_SANITIZE_STRING);
    $service_fee = floatval($data["service_fee"]);
    $charges = floatval($data["charges"]);
    $transaction_type = "deposit";

    // Calculate net amount paid
    if (!is_numeric($amount) || !is_numeric($service_fee)) {
        throw new Exception("Amount and service_fee must be numeric");
    }
    $amountPaid = round($amount - $service_fee, 2);

    // Get IP
    $ip_address = $_SERVER["REMOTE_ADDR"] ?? "Unknown";

    // Begin DB transaction
    $conn->beginTransaction();

    // Save transaction
    $stmt = $conn->prepare("
        INSERT INTO transactions 
        (uuid, currency, transaction_ref, transaction_type, account_number, amount, country, service_fee, charges, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    $stmt->execute([$uuid, $currency, $transactionRef, $transaction_type, $phone, $amountPaid, $country, $service_fee, $charges]);

    // Log audit
    $action = "User Deposit Transaction";
    $user_agent = $_SERVER["HTTP_USER_AGENT"] ?? "Unknown";
    $stmt = $conn->prepare("
        INSERT INTO audit_logs (uuid, action, user_agent, ip_address, created_at)
        VALUES (?, ?, ?, ?, NOW())
    ");
    $stmt->execute([$uuid, $action, $user_agent, $ip_address]);

    // Commit
    $conn->commit();
    logTransaction("Transaction recorded successfully: Ref $transactionRef", "success");

    // Prepare payload
    $payload = json_encode([
        "phone" => $phone,
        "amount" => $amount,
        "country" => $country,
        "currency" => $currency,
        "transactionRef" => $transactionRef,
        "otp" => $otp,
        "customer" => $customer,
    ]);
    
    logTransaction("Payload: $payload", "success");

    // cURL request
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => "https://api.eversend.co/v1/collections/momo",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => "",
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => "POST",
        CURLOPT_POSTFIELDS => $payload,
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "Authorization: Bearer {$tokenData}",
        ],
    ]);

    $response = curl_exec($curl);
    $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);

    if ($response === false) {
        throw new Exception("cURL error: " . curl_error($curl));
    }

    curl_close($curl);

    if ($http_status !== 200) {
        $status = "failed";
        $conn->beginTransaction();
        $stmt = $conn->prepare("UPDATE transactions SET status = ? WHERE transaction_ref = ?");
        $stmt->execute([$status, $transactionRef]);
        $conn->commit();

        throw new Exception("MoMo Transaction Failed. HTTP Status: $http_status, Response: $response");
    }

    logTransaction("MoMo Transaction Successful: Ref $transactionRef, Response: $response", "success");
    echo $response;

} catch (Exception $e) {
    $errorLog = "Error: " . $e->getMessage();
    logTransaction($errorLog, "error");

    if ($conn->inTransaction()) {
        $conn->rollBack();
    }

    if (isset($transactionRef)) {
        if (!$conn->inTransaction()) {
            $conn->beginTransaction();
        }
        $stmt = $conn->prepare("UPDATE transactions SET status = ? WHERE transaction_ref = ?");
        $stmt->execute(['failed', $transactionRef]);
        $conn->commit();
    }

    http_response_code(400);
    echo json_encode([
        "status" => "error",
        "message" => "MoMo transaction failed",
        "error" => $e->getMessage(),
    ]);
}

?>
