<?php
header('Content-Type: application/json');

include '../functions.php';
include '../../connect.php';
include '../../company-details.php';

// Fetch Token
$tokenData = getEversendToken();

// Allow only POST requests
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit();
}

// Get and decode JSON input
$json = file_get_contents('php://input');
$data = json_decode($json, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON data']);
    exit();
}

// Define required fields
$required_fields = ['token', 'uuid', 'phoneNumber', 'firstName', 'lastName', 'country', 'serviceFee', 'totalAmount'];
foreach ($required_fields as $field) {
    if (!isset($data[$field]) || empty(trim($data[$field]))) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => "$field is required"]);
        exit();
    }
}

// Sanitize and validate inputs
$uuid = filter_var(trim($data['uuid']), FILTER_SANITIZE_STRING);
$token = filter_var(trim($data['token']), FILTER_SANITIZE_STRING);
$phoneNumber = filter_var(trim($data['phoneNumber']), FILTER_SANITIZE_STRING);
$firstName = filter_var(trim($data['firstName']), FILTER_SANITIZE_STRING);
$lastName = filter_var(trim($data['lastName']), FILTER_SANITIZE_STRING);
$country = filter_var(trim($data['country']), FILTER_SANITIZE_STRING);
$service_fee = filter_var($data['serviceFee'], FILTER_VALIDATE_FLOAT);
$total_amount = filter_var($data['totalAmount'], FILTER_VALIDATE_FLOAT);

$transactionRef = uniqid('txn_');

// Initialize cURL request
$curl = curl_init();
curl_setopt_array($curl, [
    CURLOPT_URL => "https://api.eversend.co/v1/payouts",
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_CUSTOMREQUEST => 'POST',
    CURLOPT_POSTFIELDS => json_encode([
        'token' => $token,
        'country' => $country,
        'phoneNumber' => $phoneNumber,
        'firstName' => $firstName,
        'lastName' => $lastName,
        'transactionRef' => $transactionRef
    ]),
    CURLOPT_HTTPHEADER => [
        'Content-Type: application/json',
        "Authorization: Bearer {$tokenData}"
    ],
]);

$response = curl_exec($curl);
$http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
curl_close($curl);

if ($response === false) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Failed to connect to Eversend API']);
    exit();
}

$responseData = json_decode($response, true);
if ($http_status !== 200 || !isset($responseData['success']) || $responseData['success'] !== true) {
    http_response_code($http_status);
    echo $response;
    exit();
}

// Extract required values from Eversend response
$eversendFees = $responseData['data']['transaction']['fees'];
$country = $responseData['data']['transaction']['destinationCountry'];
$amount = $responseData['data']['transaction']['amount'];
$currency = $responseData['data']['transaction']['currency'];

// Database transaction
try {
    $conn->beginTransaction();

    // Insert transaction
    $stmt = $conn->prepare("INSERT INTO transactions (uuid, currency, transaction_ref, transaction_type, account_number, amount, country, service_fee, charges, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");
    $stmt->execute([$uuid, $currency, $transactionRef, 'withdraw', $phoneNumber, $amount, $country, $service_fee, $eversendFees, 'pending']);

    // Log audit trail
    $stmt = $conn->prepare("INSERT INTO audit_logs (uuid, action, user_agent, ip_address, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->execute([$uuid, 'Card Withdraw Transaction', $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown', $_SERVER['REMOTE_ADDR'] ?? 'Unknown']);
    
    // Deduct amount from wallet for payouts
    $stmt = $conn->prepare("UPDATE wallets SET amount = amount - ? WHERE uuid = ? AND currency = ? AND amount >= ?");
    $stmt->execute([$amount, $uuid, $currency, $amount]);
    
    // Insert commission and earnings using transaction reference from payload
    $stmt = $conn->prepare("INSERT INTO earnings (uuid, currency, transaction_ref, service_name, status, amount, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    $stmt->execute([$uuid, $currency, $transaction_ref, 'exchange', 'completed', $service_fee]);

     // Update commission total
    $updateSql = "UPDATE commission SET amount = amount + :service_fee WHERE wallet = :currency";
    $updateStmt = $conn->prepare($updateSql);
    $updateStmt->execute([':service_fee' => $service_fee, ':currency' => $currency]);

    $conn->commit();

    // Return Eversend response
    http_response_code($http_status);
    echo $response;
} catch (Exception $e) {
    $conn->rollBack();
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Transaction failed: '.$e->getMessage()]);
}
?>
